"""
Module de définition des outils pour les données de répartition des sols
"""
#liste des imports nécessaires
from requests import get
from json import loads
from csv import reader
from sqlite3 import connect
from matplotlib import pyplot

#cas particulier du réseau ENSIBS: indiquer la porte de sortie
sorties = {'http':'193.52.32.156:3128',
           'https':'193.52.32.156:3128'}

def recuperer_code_insee(nom_commune:str)->int:
    """
    Fonction pour récupérer le code insee à partir d'un  nom de commune
    En entrée: 
    - nom de la commune recherchée, type str
    Sortie:
    le code insee, type int, 0 si non trouvé
    """
    #construction de l'url de recherche
    url = f"https://public.opendatasoft.com/api/explore/v2.1/catalog/datasets/correspondance-code-cedex-code-insee/records?where=libelle%20like%20%27{nom_commune}%27&limit=20"
    #lancement de l'appel via une action GET, protocole HTTP
    contenu_api = get(url)#, proxies=sorties)
    #récupération du contenu de l'API, au format texte puis chargement JSON
    contenu_json = loads(contenu_api.text)
    #récupération de l'entrée results dans le dictionnaire obtenu
    liste_resultats = contenu_json.get('results')
    #par défaut...
    code_insee = 0
    #un résultat ?
    if liste_resultats:
        #oui, alors prenons le premier
        commune = liste_resultats[0]
        #nous devons obtenir le code insee dans le dictionnaire des informations de la commune obtenue
        code_insee = commune.get('insee')
    #nous renvoyons l'information trouvée, ou celle par défaut
    return int(code_insee)

def convertir_csv_vers_sql():
    """
    Fonction pour convertir le BDD Corine Lander (CSV) en SQL
    A appeler une seule fois !
    En entrée: 
    - Rien
    Sortie:
    Rien
    """
    #ouverture du fichier CSV, qui doit être à la racine du projet
    with open('clc_etat_com_n1.csv', 'r', encoding="utf-8") as fichier_csv:
        #création d'un parseur CSV
        parseur_csv = reader(fichier_csv, delimiter=";")
        #création d'un liste pour le stockage des lignes correctes
        lignes = []
        #parcours du CSV
        for ligne in parseur_csv:
            #contrôle: est ce une bonne ligne ?
            if ligne and ligne[0].isdigit():
                #oui, alors je retire la colonne du nom de la base qui ne m'intéresse pas
                ligne.pop(2)
                #ajout de toutes les autres colonnes dans le stockage des lignes
                lignes.append(ligne)
    #a ce stade, j'ai en mémoire toutes les répartitions connues
    #création de la base de données
    #attention: si le fichier existe déjà, le code plante lors de la création de la table!
    ma_bdd = connect('bdd.corine.sqlite3')
    #création du curseur pour intéraction avec la BDD
    curseur = ma_bdd.cursor()
    #création de la requete SQL pour créer la table des répartitions
    curseur.execute("CREATE TABLE repartition (insee numeric, annee numeric, cl1 real, cl2 real, cl3 real, cl4 real, cl5 real)")
    #injection de toutes les répartitions dans la table SQL
    curseur.executemany("INSERT INTO repartition VALUES (?,?,?,?,?,?,?)", lignes)
    #validation des commandes SQL
    ma_bdd.commit()
    #fermeture de la base
    ma_bdd.close()

def afficher_repartition(code_insee:int)->None:
    """
    Fonction afficher la répartition des sols d'une commune via une image png !
    En entrée: 
    - le code insee de la commune, type int
    Sortie:
    Rien
    """
    #connexion à la base de données
    with connect('bdd.corine.sqlite3') as ma_bdd:
        #création du curseur pour la lecture
        curseur = ma_bdd.cursor()
        #requete SQL pour récupérer la répartition d'une commune
        lignes = curseur.execute(f"SELECT * FROM repartition WHERE insee = {code_insee}")
        #chargement en mémoire du résultat
        lignes_en_memoire = lignes.fetchall()
        #création d'un figure, avec autant de graphes que de lignes (ie, d'année)
        fig, graphes = pyplot.subplots(nrows=len(lignes_en_memoire))
        #pour chaque ligne, ie, année
        for position, ligne in enumerate(lignes_en_memoire):
            #titre du graphe
            titre = f"Année: {ligne[1]}"
            #liste des valeurs
            valeurs = ligne[2:]
            #légendes
            etiquettes = ('I','II','III','IV','V')
            #création du graphique
            graphes[position].pie(valeurs, labels=etiquettes, textprops={'fontsize':4})
            #positionnement du titre
            graphes[position].set_title(titre, fontsize=4)
        #ici, pas d'affichage: sauvegarde du fichier au format png
        pyplot.savefig('repartition.png', dpi=400)

if __name__ == "__main__":
    #programme principal, en 3 étapes:
    #récupération du nom de la ville
    nom_ville = input("Saisie la ville: ")
    #récupération de son code insee
    retour = recuperer_code_insee(nom_ville)
    if retour != 0:
        #ici, à faire 1 seule fois; code optimisable !
        #convertir_csv_vers_sql()
        #obtenir le rendu
        afficher_repartition(retour)
    else:
        print("Code insee non trouvé pour la commune")


